/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.core.request.resource;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.ArgumentMatchers.same;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;

import jakarta.servlet.http.HttpServletResponse;

import org.apache.wicket.request.http.WebRequest;
import org.apache.wicket.request.http.WebResponse;
import org.apache.wicket.request.resource.ByteArrayResource;
import org.apache.wicket.request.resource.IResource.Attributes;
import org.apache.wicket.util.tester.WicketTestCase;
import org.junit.jupiter.api.Test;

/**
 * Unit tests for {@link ByteArrayResource}
 */
class ByteArrayResourceTest extends WicketTestCase
{

	/**
	 * Unit test for {@link ByteArrayResource} with static byte array.
	 */
	@Test
	void staticResource()
	{
		String contentType = "application/octet-stream";
		byte[] array = new byte[] { 1, 2, 3 };
		ByteArrayResource resource = new ByteArrayResource(contentType, array)
		{
			private static final long serialVersionUID = 1L;

			@Override
			protected void configureCache(ResourceResponse data, Attributes attributes)
			{
				// no caching is needed
			}
		};

		WebRequest request = mock(WebRequest.class);
		WebResponse response = mock(WebResponse.class);

		Attributes attributes = new Attributes(request, response);
		resource.respond(attributes);

		verify(response).write(same(array));
		verify(response).setContentLength(eq(3L));
		verify(response).setContentType(eq(contentType));
	}

	/**
	 * Unit test for {@link ByteArrayResource} with dynamically generated byte array.
	 */
	@Test
	void dynamicResource()
	{
		String contentType = "application/octet-stream";
		final byte[] array = new byte[] { 1, 2, 3 };
		ByteArrayResource resource = new ByteArrayResource(contentType)
		{
			private static final long serialVersionUID = 1L;

			@Override
			protected byte[] getData(Attributes attributes)
			{
				return array;
			}

			@Override
			protected void configureCache(ResourceResponse data, Attributes attributes)
			{
				// no caching is needed
			}
		};

		WebRequest request = mock(WebRequest.class);
		WebResponse response = mock(WebResponse.class);

		Attributes attributes = new Attributes(request, response);
		resource.respond(attributes);

		verify(response).write(same(array));
		verify(response).setContentLength(eq(3L));
		verify(response).setContentType(eq(contentType));
	}

	/**
	 * Content-Type should be resolved in the following way:
	 * <ol>
	 * <li>first check if the user passed it</li>
	 * <li>fallback to check by filename extension</li>
	 * <li>as final resort always use 'application/octet-stream'</li>
	 * </ol>
	 * 
	 * See https://issues.apache.org/jira/browse/WICKET-4119
	 */
	@Test
	void contentType()
	{
		String userSpecifiedContentType = "text/custom";
		ByteArrayResource userSpecified = new ByteArrayResource(userSpecifiedContentType,
			new byte[] { 1, 2, 3 });

		tester.startResource(userSpecified);
		assertEquals(userSpecifiedContentType, tester.getLastResponse().getContentType());

		ByteArrayResource resolvedByExtension = new ByteArrayResource(null, new byte[] { 1, 2, 3 },
			"image.png");

		tester.startResource(resolvedByExtension);
		assertEquals("image/png", tester.getLastResponse().getContentType());

		ByteArrayResource finalResortOctetStream = new ByteArrayResource(null,
			new byte[] { 1, 2, 3 }, null);

		tester.startResource(finalResortOctetStream);
		assertEquals("application/octet-stream", tester.getLastResponse().getContentType());
	}

	/**
	 * https://issues.apache.org/jira/browse/WICKET-4301
	 *
	 * Verify that HttpServletResponse#SC_NOT_FOUND is returned if there is no data.
	 */
	@Test
	void noData() {
		ByteArrayResource resource = new ByteArrayResource("text/plain");
		tester.startResource(resource);

		assertEquals(HttpServletResponse.SC_NOT_FOUND, tester.getLastResponse().getStatus());
		assertEquals(0, tester.getLastResponse().getBinaryContent().length);
	}
}
