/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.apache.skywalking.apm.plugin.spring.cloud.gateway.v412x;

import org.apache.skywalking.apm.agent.core.context.CarrierItem;
import org.apache.skywalking.apm.agent.core.context.ContextCarrier;
import org.apache.skywalking.apm.agent.core.context.ContextManager;
import org.apache.skywalking.apm.agent.core.context.ContextSnapshot;
import org.apache.skywalking.apm.agent.core.context.tag.Tags;
import org.apache.skywalking.apm.agent.core.context.trace.AbstractSpan;
import org.apache.skywalking.apm.agent.core.context.trace.SpanLayer;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.EnhancedInstance;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.InstanceMethodsAroundInterceptor;
import org.apache.skywalking.apm.agent.core.plugin.interceptor.enhance.MethodInterceptResult;
import org.apache.skywalking.apm.plugin.spring.cloud.gateway.v4x.define.EnhanceObjectCache;
import org.apache.skywalking.apm.util.StringUtil;
import org.reactivestreams.Publisher;
import reactor.netty.NettyOutbound;
import reactor.netty.http.client.HttpClientRequest;

import java.lang.reflect.Method;
import java.net.URL;
import java.util.function.BiFunction;

import static org.apache.skywalking.apm.network.trace.component.ComponentsDefine.SPRING_CLOUD_GATEWAY;

/**
 * This class intercept <code>send</code> method.
 * <p>
 * In before method, create a new BiFunction lambda expression for setting <code>ContextCarrier</code> to http header
 * and replace the original lambda in argument
 */
public class HttpClientFinalizerSendV412Interceptor implements InstanceMethodsAroundInterceptor {

    @Override
    public void beforeMethod(EnhancedInstance objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes,
                             MethodInterceptResult result) throws Throwable {
        EnhanceObjectCache enhanceObjectCache = (EnhanceObjectCache) objInst.getSkyWalkingDynamicField();
        if (enhanceObjectCache == null) {
            return;
        }
        ContextSnapshot contextSnapshot = enhanceObjectCache.getContextSnapshot();
        if (contextSnapshot == null) {
            return;
        }
        AbstractSpan span = ContextManager.createLocalSpan("SpringCloudGateway/send");
        ContextManager.continued(contextSnapshot);
        span.setComponent(SPRING_CLOUD_GATEWAY);
        span.prepareForAsync();

        if (StringUtil.isNotEmpty(enhanceObjectCache.getUrl())) {
            URL url = new URL(enhanceObjectCache.getUrl());

            ContextCarrier contextCarrier = new ContextCarrier();
            AbstractSpan abstractSpan = ContextManager.createExitSpan(
                    "SpringCloudGateway/sendRequest", contextCarrier, getPeer(url));
            Tags.URL.set(abstractSpan, enhanceObjectCache.getUrl());
            abstractSpan.prepareForAsync();
            abstractSpan.setComponent(SPRING_CLOUD_GATEWAY);
            abstractSpan.setLayer(SpanLayer.HTTP);
            ContextManager.stopSpan(abstractSpan);

            BiFunction<? super HttpClientRequest, ? super NettyOutbound, ? extends Publisher<Void>> finalSender = (BiFunction<? super HttpClientRequest, ? super NettyOutbound, ? extends Publisher<Void>>) allArguments[0];
            allArguments[0] = (BiFunction<HttpClientRequest, NettyOutbound, Publisher<Void>>) (request, outbound) -> {
                Publisher publisher = finalSender.apply(request, outbound);

                CarrierItem next = contextCarrier.items();
                while (next.hasNext()) {
                    next = next.next();
                    request.requestHeaders().remove(next.getHeadKey());
                    request.requestHeaders().set(next.getHeadKey(), next.getHeadValue());
                }
                return publisher;
            };
            enhanceObjectCache.setCacheSpan(abstractSpan);
        }
        ContextManager.stopSpan(span);
        enhanceObjectCache.setSpan1(span);
    }

    private String getPeer(URL url) {
        return url.getHost() + ":" + url.getPort();
    }

    @Override
    public Object afterMethod(EnhancedInstance objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes,
                              Object ret) {
        ((EnhancedInstance) ret).setSkyWalkingDynamicField(objInst.getSkyWalkingDynamicField());
        return ret;
    }

    @Override
    public void handleMethodException(EnhancedInstance objInst, Method method, Object[] allArguments,
                                      Class<?>[] argumentsTypes, Throwable t) {

    }
}
