/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.polaris.core.policy.validator;

import org.apache.polaris.core.entity.PolarisEntitySubType;
import org.apache.polaris.core.entity.PolarisEntityType;

/** Validates and parses a given policy content string against its defined schema. */
public interface PolicyValidator {

  /**
   * Validates the provided policy content.
   *
   * @param content the policy content to parse and validate
   * @throws InvalidPolicyException if the content is not valid
   */
  void validate(String content) throws InvalidPolicyException;

  /**
   * Determines whether the policy is attachable to a target entity.
   *
   * <p>This method examines the provided {@link PolarisEntityType} and {@link PolarisEntitySubType}
   * to decide if a policy is applicable for attachment to the target entity.
   *
   * @param entityType the type of the target entity
   * @param entitySubType the subtype of the target entity
   * @return {@code true} if the policy can be attached to the target entity; {@code false}
   *     otherwise
   */
  boolean canAttach(PolarisEntityType entityType, PolarisEntitySubType entitySubType);
}
